package cz.geek.banking.fio;

import cz.geek.banking.BankStatement;
import cz.geek.banking.BankStatementImporterException;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import jakarta.mail.BodyPart;
import jakarta.mail.MessagingException;
import jakarta.mail.Part;
import jakarta.mail.Session;
import jakarta.mail.internet.MimeMessage;
import jakarta.mail.internet.MimeMultipart;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.time.LocalDate;
import java.util.*;

/**
 * Importer is not synchronized. It is recommended to create separate importer instance for each thread.
 * If multiple threads access a importer concurrently, it must be synchronized externally.
 */
public class FioImporter {

	private final Logger log = LoggerFactory.getLogger(getClass());

	private final Session session;
	private final DecimalFormat fmt;

	public FioImporter() {
		session = Session.getDefaultInstance(new Properties());
		fmt = (DecimalFormat) NumberFormat.getInstance(new Locale("cs"));
		fmt.setParseBigDecimal(true);
	}

	public BankStatement parse(InputStream is) {
		log.info("Processing message");
		Assert.notNull(is, "Input stream must not be null");
		try {
			MimeMessage msg = new MimeMessage(session, is);
			return parse(msg);
		} catch (MessagingException e) {
			throw new BankStatementImporterException("Unable to parse message", e);
		} catch (IOException e) {
			throw new BankStatementImporterException("Unable to read message content", e);
		} finally {
			try {
				is.close();
			} catch (IOException ignored) { }
		}
	}

	private BankStatement parse(Part part) throws IOException, MessagingException {
		if (part.getContent() instanceof String) {
			String content = (String) part.getContent();
			return doParse(content);
		}
		if (part.getContent() instanceof MimeMultipart) {
			MimeMultipart content = (MimeMultipart) part.getContent();
			if (content.getCount() == 1) {
				BodyPart bodyPart = content.getBodyPart(0);
				return parse(bodyPart);
			}
			throw new BankStatementImporterException("Invalid number of parts: " + content.getCount());
		}
		throw new BankStatementImporterException("Unknown part " + part.getContent().getClass());
	}

	public BankStatement parse(String email) {
		Assert.notNull(email, "Input must not be null");
		return parse(new ByteArrayInputStream(email.getBytes(Charset.forName("UTF-8"))));
	}

	protected BankStatement doParse(String content) {
		log.debug("Message content:\n {}", content);
		String note;
		boolean negate = false;
		Map<String, String> values = parseContent(content);
		String value = StringUtils.deleteWhitespace(values.get("Částka"));
		if (values.get("Výdaj na kontě") != null) {
			note = values.get("US");
			negate = true;
		} else {
			note = values.get("Zpráva příjemci");
		}
		note = StringUtils.strip(note);
		try {
			BigDecimal amount = (BigDecimal) fmt.parse(value);
			if (negate)
				amount = amount.negate();
			String vs = values.get("VS");
			Long v = StringUtils.isNotBlank(vs) ? Long.valueOf(StringUtils.strip(vs)) : null;
			BankStatement bs = new BankStatement(LocalDate.now(), v, values.get("SS"), values.get("KS"), amount, note);
			log.debug("Parsed {}", bs);
			return bs;
		} catch (ParseException e) {
			throw new BankStatementImporterException("Unable to parse '" + value + "' at position " + e.getErrorOffset(), e);
		}
	}

	private Map<String, String> parseContent(String content) {
		Map<String,String> values = new LinkedHashMap<String, String>();
		StringTokenizer st = new StringTokenizer(content, "\n");
		int lines = 0;
		while (st.hasMoreTokens()) {
			lines++;
			String line = st.nextToken();
			String[] split = line.split(": ", 2);
			switch (split.length) {
				case 1: values.put(split[0], null); break;
				case 2: values.put(split[0], split[1]); break;
				default: log.warn("Skipping line {}: {}", lines, line); break;
			}
		}
		return values;
	}

}
