package cz.geek.gdata.client;

import com.google.gdata.client.GoogleService;
import com.google.gdata.client.Service;
import com.google.gdata.util.AuthenticationException;

import cz.geek.cache.Cache;

/**
 * Cache and auth factory
 * @author martin
 */
public class ServiceFactory {

	/**
	 * Set authentication info for given {@link GoogleService}
	 * @param <T> extends GoogleService
	 * @param service ordinary service
	 * @param username user name
	 * @param password user password
	 * @return authenticated service or original service if no auth info was specified
	 * @throws AuthenticationException 
	 * @see GoogleService#setUserCredentials(String, String)
	 */
	public static <T extends GoogleService> T getService(T service, String username, String password) throws AuthenticationException {
		service.setUserCredentials(username, password);
		return service;
	}

	/**
	 * Set session token for given {@link GoogleService}
	 * @param <T> extends GoogleService
	 * @param service ordinary service
	 * @param authSubToken session token
	 * @return authenticated service
	 * @throws AuthenticationException 
	 * @see GoogleService#setAuthSubToken(String)
	 */
	public static <T extends GoogleService> T getService(T service, String authSubToken) throws AuthenticationException {
		service.setAuthSubToken(authSubToken);
		return service;
	}

	/**
	 * Decorate given {@link Service} with caching
	 * @param <T> extends Service
	 * @param service ordinary service
	 * @param cache desired cache
	 * @return decorated cached service
	 */
	public static <T extends Service> T getService(T service, Cache<String, CachedResponse> cache) {
		service.setRequestFactory(new CachingRequestFactory(cache));
		return service;
	}
	
	/**
	 * Set auth info and decorate with cache in one shot
	 * @param <T>
	 * @param service ordinary service
	 * @param cache desired cache
	 * @param username user name
	 * @param password user password
	 * @return decorated cached and authenticated service
	 * @throws AuthenticationException
	 * @see #getService(Service, Cache)
	 * @see #getService(GoogleService, String, String)
	 */
	public static <T extends GoogleService> T getService(T service, Cache<String, CachedResponse> cache, String username, String password) throws AuthenticationException {
		T cached = getService(service, cache);
		T auth = getService(cached, username, password);
		return auth;
	}

	/**
	 * Set session token and decorate with cache in one shot
	 * @param <T>
	 * @param service ordinary service
	 * @param cache desired cache
	 * @param authSubToken session token
	 * @return decorated cached and authenticated service
	 * @throws AuthenticationException
	 * @see #getService(Service, Cache)
	 * @see #getService(GoogleService, String, String)
	 */
	public static <T extends GoogleService> T getService(T service, Cache<String, CachedResponse> cache, String authSubToken) throws AuthenticationException {
		T cached = getService(service, cache);
		T auth = getService(cached, authSubToken);
		return auth;
	}

}
