package cz.geek.gdata.client;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gdata.client.http.GoogleGDataRequest;
import com.google.gdata.client.http.HttpAuthToken;
import com.google.gdata.client.http.HttpUrlConnectionSource;
import com.google.gdata.data.DateTime;
import com.google.gdata.util.ContentType;
import com.google.gdata.util.ServiceException;

import cz.geek.cache.Cache;

public class CachedRequest extends GoogleGDataRequest {
	
	protected Cache<String, CachedResponse> cache;
	
	protected CachedResponse resp;
	
	private final Logger log = LoggerFactory.getLogger(this.getClass());

	protected CachedRequest(Cache<String, CachedResponse> cache, RequestType type, URL requestUrl, ContentType contentType, HttpAuthToken authToken, Map<String, String> headerMap,
			Map<String, String> privateHeaderMap, HttpUrlConnectionSource connectionSource) throws IOException {
		super(type, requestUrl, contentType, authToken, headerMap, privateHeaderMap, connectionSource);
		this.cache = cache;
	}

	/** 
	 * {@inheritDoc}
	 * Skips if request is in cache.
	 */
	@Override
	public void execute() throws IOException, ServiceException {
		String url = requestUrl.toString();
		if (!cache.containsKey(url)) {
			log.debug("execute()");
			super.execute();
			log.info("Storing {} to cache", url);
			InputStream in = super.getResponseStream();
			Map<String, List<String>> headers = httpConn.getHeaderFields();
			resp = new CachedResponse(in, headers);
			cache.put(url, resp);
		} else {
			log.debug("execute() skipping");
			resp = cache.get(url);
		}
	}

	@Override
	public InputStream getResponseStream() throws IOException {
		String url = requestUrl.toString();
		log.info("getResponseStream() {} from cache", url);
		return resp.getResponseStream();
	}

	@Override
	public String getResponseHeader(String headerName) {
		String url = requestUrl.toString();
		log.info("getResponseHeader() {} from cache", url);
		String value = resp.getHeaderField(headerName);
		return value;
	}

	@Override
	public ContentType getResponseContentType() {
		String value = getResponseHeader("Content-Type");
		if (value == null)
			return null;
		return new ContentType(value);
	}

	@Override
	public DateTime getResponseDateHeader(String headerName) {
		String url = requestUrl.toString();
		log.info("getResponseDateHeader() {} from cache", url);
		long dateValue = resp.getHeaderFieldDate(headerName, -1);
	    return (dateValue >= 0) ? new DateTime(dateValue) : null;
	}

}
